<?php
/**
 * Add Link Tag
 *
 * @package     AutomatorWP\Integrations\ShortLinksPro\Actions\Add_Link_Tag
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_ShortLinksPro_Add_Link_Tag extends AutomatorWP_Integration_Action {

    public $integration = 'shortlinkspro';
    public $action = 'shortlinkspro_add_link_tag';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add tag to a link', 'automatorwp-pro' ),
            'select_option'     => __( 'Add <strong>tag</strong> to a link', 'automatorwp-pro' ),
            /* translators: %1$s: Tag title. %2$s: Link title */
            'edit_label'        => sprintf( __( 'Add %1$s to %2$s link', 'automatorwp-pro' ), '{tag}', '{link}' ),
            /* translators: %1$s: Tag title. %2$s: Link title */
            'log_label'         => sprintf( __( 'Add %1$s to %2$s link', 'automatorwp-pro' ), '{tag}', '{link}' ),
            'options'           => array(
                'link' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'link',
                    'name'              => __( 'Link:', 'automatorwp-pro' ),
                    'option_default'    => __( 'Select a link', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_shortlinkspro_get_links',
                    'options_cb'        => 'automatorwp_shortlinkspro_options_cb_link',
                    'default'           => ''
                ) ),
                'tag' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'tag',
                    'name'              => __( 'Tag:', 'automatorwp-pro' ),
                    'option_default'    => __( 'Select a tag', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_shortlinkspro_get_tags',
                    'options_cb'        => 'automatorwp_shortlinkspro_options_cb_tag',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {
    
        $this->result = '';
        $terms = array();
        
        // Shorthand
        $link_id = $action_options['link'];
        $tag_id = $action_options['tag']; 
        $table = 'shortlinkspro_link_tags';

        // Bail if no target URL
        if ( empty( $link_id ) ) {
            $this->result = __('No link selected.', 'automatorwp-pro');
            return;
        }

        // Bail if no tag
        if ( empty( $tag_id ) ) {
            $this->result = __('No tag selected.', 'automatorwp-pro');
            return;
        }

        // Get the current tags related to lonk
        $link_tags = automatorwp_shortlinkspro_get_tags_from_link( $link_id );
        
        foreach ( $link_tags as $tag ) {
            $terms[] = $tag->id;
        }

        // Add tag if not exists
        if ( ! in_array( $tag_id, $terms ) ) {
            $terms[] = $tag_id;
        } else {
            $this->result = __('The link already has the tag.', 'automatorwp-pro');
            return;
        }
        
        shortlinkspro_handle_terms_save( $link_id, $terms, $table );

        $this->result = __('Link tag added.', 'automatorwp-pro');
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_ShortLinksPro_Add_Link_Tag();